/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::electrochemistryModel

Description
    Solves the electrochemistry including boundary conditions for consumed
    and produced species, voltage and current density.

\*---------------------------------------------------------------------------*/

#include "electrochemistryModel.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::electrochemistryModel::electrochemistryModel
(
    patchDatabase& pm,
    materialDatabase& matAir,
    materialDatabase& matFuel,
    flowBalance& airBalance,
    flowBalance& fuelBalance,
    electrolyteModel& elec,
    const ohmicOverpotentialModel& ohm,
    const activationOverpotentialModel& airActivationOverpotential,
    const activationOverpotentialModel& fuelActivationOverpotential,
    mappingModel& map,
    IOdictionary& cellProperties,
    volScalarField& Tcell
)
:
    runTime_(matAir.mesh().time()),
    electrolyteMesh_(elec.mesh()),
    airMesh_(matAir.mesh()),
    fuelMesh_(matFuel.mesh()),
    pm_(pm),
    matAir_(matAir),
    matFuel_(matFuel),
    airBalance_(airBalance),
    fuelBalance_(fuelBalance),
    elec_(elec),
    ohm_(ohm),
    airActivationOverpotential_(airActivationOverpotential),
    fuelActivationOverpotential_(fuelActivationOverpotential),
    map_(map),
    rxnDict_
    (
        IOobject
        (
            "rxnProperties",
            runTime_.constant(),
            Tcell.mesh(),
            IOobject::MUST_READ,
            IOobject::NO_WRITE
        )
    ),
    rxnSpCoef_(rxnDict_.lookup("rxnSpecies")),
    fluxMolarMass_(-1),
    fluxElectrons_(-1),
    activationFactor_(cellProperties.lookupOrDefault<scalar>("activationFactor", 1.0)),
    PreIterAct_(readScalar(cellProperties.lookup("PreIterAct"))),
    relaxCurrent_(readScalar(cellProperties.lookup("relaxJ"))),
    relaxMem_(cellProperties.lookupOrDefault<scalar>("relaxMembrane", 1.0)),
    numCells_(readScalar(cellProperties.lookup("numCells"))),
    currentSign_(cellProperties.lookupOrDefault<scalar>("currentSign", -1.0)),
    Rhat_(0.0),
    V_("voltage", dimensionSet(1,2,-3,0,0,-1,0), cellProperties),
    ibar0_("ibar0", dimensionSet(0,-2,0,0,0,1,0), cellProperties),
    galvanostatic_(cellProperties.lookup("galvanostatic")),
    firstTime_(true),
    i_(pm_.anodePatch().size()),
    iOld_(pm_.anodePatch().size()),
    iNew_(pm_.anodePatch().size()),
    anodeT_(pm_.electrolyteAnodePatch().size()),
    cathodeT_(pm_.electrolyteCathodePatch().size()),
    Nernst_(pm_.anodePatch().size()),
    mfluxOld_(pm.cathodePatch().size(), 0.0),
    xAir_(matAir_.species().size()),
    xFuel_(matFuel_.species().size()),
    voltage_
    (
        IOobject
        (
            "V",
            runTime_.timeName(),
            electrolyteMesh_,
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        electrolyteMesh_,
        V_,
        zeroGradientFvPatchScalarField::typeName
    ),
    NernstPot_
    (
        IOobject
        (
            "NernstPot",
            runTime_.timeName(),
            electrolyteMesh_,
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        electrolyteMesh_,
        1,
        zeroGradientFvPatchScalarField::typeName
    ),
    etaA_
    (
        IOobject
        (
            "etaA",
            runTime_.timeName(),
            electrolyteMesh_,
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        electrolyteMesh_,
        dimensionedScalar
        (
            "etaA",
            dimensionSet(1,2,-3,0,0,-1,0),
            0
        ),
        zeroGradientFvPatchScalarField::typeName
    ),
    etaC_
    (
        IOobject
        (
            "etaC",
            runTime_.timeName(),
            electrolyteMesh_,
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        electrolyteMesh_,
        dimensionedScalar
        (
            "etaC",
            dimensionSet(1,2,-3,0,0,-1,0),
            0
        ),
        zeroGradientFvPatchScalarField::typeName
    ),
    idensity_
    (
        IOobject
        (
            "i",
            runTime_.timeName(),
            electrolyteMesh_,
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        electrolyteMesh_,
        dimensionedScalar
        (
            "i",
            dimensionSet(0,-2,0,0,0,1,0),
            ibar0_.value()
        ),
        zeroGradientFvPatchScalarField::typeName
    ),
    electrochemicalHeating_
    (
        IOobject
        (
            "electrochemicalHeating",
            runTime_.timeName(),
            electrolyteMesh_,
            IOobject::READ_IF_PRESENT,
            IOobject::AUTO_WRITE
        ),
        electrolyteMesh_,
        0,
        zeroGradientFvPatchScalarField::typeName
    ),
    Tcell_(Tcell),
    Uair_(matAir.U()),
    Ufuel_(matFuel.U()),
    Yair_(matAir.Y()),
    Yfuel_(matFuel.Y())
{
    // it is very important, where the variables are defined!!!
    // mass/molar fluxes: in air/fuel anode/cathode patch
    // i, iOld_, iNew_: defined by mass flux in anode=fuel=anodePatch
    // temperatures (anodeT_, cathodeT_): in air/fuel
    // water flux through electrolyte: inside electrolyte
    // electrochemical heating: at anode (fuel) patch
    // Nernst_: all at anode (fuel) patch
    // ASR: all at anode (fuel) patch

    // activation overpotential: defined on air/fuel cathode/anode patch

    forAll(matAir_.species(), s)
    {
        xAir_.set // molar fraction at anodePatch i.e. fuel_to_electrolyte
        (
            s,
            new scalarField(anodeT_.size(), 0)
        );
    }
    forAll(matFuel_.species(), s)
    {
        xFuel_.set // molar fraction at anodePatch i.e. fuel_to_electrolyte
        (
            s,
            new scalarField(anodeT_.size(), 0)
        );
    }

    Info<< nl << "Reading rxnProperties" << nl;
    Info<< "    rxnSpCoef contents: " << rxnSpCoef_.toc() << nl;

    Info<< "    rxnSpecies stoichiometric coefficients: " << nl;

    forAll(matAir.species(), i)
    {
        if(rxnSpCoef_.found(matAir.species()[i].name()))
        {
            Info<< "    stoichCoeff "
                << matAir.species()[i].name()
                << " = "
                << rxnSpCoef_[matAir.species()[i].name()]
                << nl;
        }
    }

    forAll(matFuel.species(), i)
    {
        if(rxnSpCoef_.found(matFuel.species()[i].name()))
        {
            Info<< "    stoichCoeff "
                << matFuel.species()[i].name()
                << " = "
                << rxnSpCoef_[matFuel.species()[i].name()]
                << nl;
        }
    }

    if(rxnSpCoef_.found("e"))
    {
        Info<< "    stoichCoeff e = " << rxnSpCoef_["e"] << nl;
    }
    else
    {
        Info<< "*** ERROR *** required rxnSpCoef_f 'e' not found" << nl;
    }

    Info<< endl;


    // the current is later computed using the consumed species in the fuel
    // find out which is the species, molar mass and number of electrons
    int fluxSpecies = -1;
    forAll(matFuel_.species(), s)
    {
        if(matFuel_.species()[s].rSign() < 0)
        {
            if(fluxSpecies < 0)
            {
                fluxSpecies = s;
            }
            else
            {
                FatalErrorIn("Create species")
                << "Only one species can be consumed in the fuel"
                << abort(FatalError);
            }
        }
    }
    if(fluxSpecies < 0)
    {
        FatalErrorIn("Create species")
           << "One species needs to be consumed in the fuel"
           << abort(FatalError);
    }
    fluxMolarMass_ = matFuel_.species()[fluxSpecies].MW();
    fluxElectrons_ = matFuel_.species()[fluxSpecies].ne();

    Info << "Using the species " << matFuel_.species()[fluxSpecies].name()
         << " with molar mass " << fluxMolarMass_
         << " and " << fluxElectrons_ << " electrons"
         << " for current calculation." << endl;


    // echo inputs from dictionary to stdout
    // -------------------------------------
    if (galvanostatic_)
    {
        Info<< "Galvanostatic run" << nl;
        Info<< "    V     = " << V_ << nl;
        Info<< "    ibar0 = " << ibar0_ << nl;

        Rhat_ = readScalar(cellProperties.lookup("Rhat"));
        Info<< "    Rhat  = " << Rhat_ << nl;
    }
    else
    {
        Info<< "Potentiostatic run" << nl;
        Info<< "    ibar0 = " << ibar0_ << nl;
        Info<< "    V     = " << V_ << nl;
    }

}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::electrochemistryModel::solve()
{
    Info << nl << "Solving electrochemistry" << endl;

    getTemperature(); // map temperature to anode and cathode patch
    calcMoleFractions();
    calcNernst();
    calcCurrentDensity();
    correctVoltage(); // calculate mean current density
    calcElectrochemicalHeating();
    // calculate water flux between anode and cathode
    elec_.solveWaterFlux(idensity_);
    updateBoundaryConditions();
    if(firstTime_)
    {
	firstTime_ = false;
    }
}

void electrochemistryModel::calcCurrentDensity()
{

    // area specific resistance, R
    // scalarField R is based on anodeT_

    // Takes voltage from boundary face and applies it at cell center of cell adjacent to boundary face
    forAll(pm_.electrolyteAnodePatch(), facei)
    {
        voltage_.boundaryFieldRef()[pm_.electrolyteAnodeID()][facei] = 
	    voltage_[pm_.electrolyteAnodePatch().faceCells()[facei]];
    }

    // mfluxOld_ is the consumed species in the fuel (see createAirSpecies.H):
    // H2 in a PEM cell
    // H2 in the hydrogen pump
    // H2 in the solid oxid fuel cell

    // direction of the current (positive charge) in electrolyte is defined as:
    // minus: anode -> cathode
    // plus: cathode -> anode

    iOld_ = currentSign_*fluxElectrons_*FF_*mfluxOld_/fluxMolarMass_;

    if (runTime_.value() <= 1) iOld_ = ibar0_.value(); //First time step

    // solve current
    // Add etaA and etaC into the current function after the pre-iteration

    // the area specific resistance is associated with the ohmic voltage_LOSS
    // therefore we use the ohmic voltage LOSS (Nernst - cell potential - other overpotentials)
    scalarField R = ohm_.ASR(anodeT_,elec_.lambda());

    if (runTime_.value() > PreIterAct_)
    {
        iNew_ = (Nernst_ - voltage_.boundaryField()[pm_.electrolyteAnodeID()] 
	    - etaA_.boundaryField()[pm_.electrolyteAnodeID()] 
	    - etaC_.boundaryField()[pm_.electrolyteAnodeID()])/R;
    }
    else
    {
        iNew_ = (Nernst_ - voltage_.boundaryField()[pm_.electrolyteAnodeID()])/R;
    }
    Info << "etaOhmic = " << gAverage(iNew_*R) << endl;

    check("iNew", iNew_, -GREAT, GREAT);
    check("iOld", iOld_, -GREAT, GREAT);

    // relax current density
    i_ = relaxCurrent_*iNew_ + (1.0 - relaxCurrent_)*iOld_;
    iOld_ = i_;

    // ensure i > 0
    i_ = Foam::max(i_, Foam::doubleScalarSMALL);
    check("i", i_, SMALL, GREAT);

    // scalarField etaAnode is computed from an inverse Butler-Volmer eqn.
    if (runTime_.value() > PreIterAct_)
    {
        calcActivationOverpotential();
    }

    // Current density
    // i and Nernst are defined on anode (fuel) patch

    // idensity and NernstPot are needed inside electrolyte

    const labelUList& aOwner = 
	electrolyteMesh_.boundary()[pm_.electrolyteAnodeID()].faceCells();
    scalarField iAnodeElectrolyte = 
	pm_.fuelAnodeToElectrolyteAnode().faceInterpolate(i_);

    forAll(iAnodeElectrolyte,facei)
    {
        idensity_[aOwner[facei]] = iAnodeElectrolyte[facei];
    }
    idensity_.correctBoundaryConditions();
    check(idensity_, -GREAT, GREAT, false);


    // Nernst Potential
    scalarField NernstAnodeElectrolyte = 
	pm_.fuelAnodeToElectrolyteAnode().faceInterpolate(Nernst_);
    forAll(NernstAnodeElectrolyte,facei)
    {
        NernstPot_[aOwner[facei]] = NernstAnodeElectrolyte[facei];
    }
    NernstPot_.correctBoundaryConditions();
}


void electrochemistryModel::calcNernst()
{
    // Nernst_ Eqn
    // E = E0 - RT/(zF) ln(Q)
    // where
    //    E = cell potential
    //    E0 = standard cell potential at temperature of interest, T
    //       = -deltaG_rxn/(zf)
    //         where deltaG_rxn = Gibbs free energy of reaction at temperature T
    //    R = universal gas constant
    //    F = Faraday constant
    //    Q = reaction quotient
    //    z = moles of electrons transferred
    //
    // for reaction Sum(ai Ri) = Sum(bj Pj)
    //
    //    Q = Qrxn = product([Pj]^bj)/product([Ri]^ai)
    //               where [X] denotes moleFraction of X
    //
    //    deltaG_rxn = deltaH_rxn - T*deltaS_rxn
    //
    //        deltaH_rxn = sum(bj*deltaH(Pj)) - sum(ai*deltaH(Ri)
    //
    //            deltaH(X) = deltaH_f^0(X) + intgral_T0^T(Cp_X(T)dT)
    //
    //        deltaS_rxn = sum(bj*S(Pj)) - sum(ai*S(Ri))
    //
    //            S(X) = S^0(X) + intgral_T0^T(Cp_X(T)/T dT)
    //
    // eg for rxn
    //                H2 + (1/2)O2 = H2O
    // at T = 1000K,
    // using tabulated gas phase thermochemistry data from NIST chemistry web book
    // http://webbook.nist.gov/chemistry/ (accessed 2012 july 10)
    // we find
    //                deltaH_rxn = -247.86 kJ/mol  (ie, per mole of H2)
    //                deltaS_rxn = -0.0553 kJ/(mol K)
    //                deltaG_rxn = deltaH_rxn - T*deltaS_rxn = -192.56 kJ/mol

    // initialize Qrxn, deltaHrxn, deltaSrxn
    scalarField Qrxn(anodeT_.size(), 1);         // reaction quotient
    scalarField deltaHrxn(anodeT_.size(), 0);    // change of enthalpie during reaction
    scalarField deltaSrxn(anodeT_.size(), 0);    // change of entropy during reaction

    // update with fuel reactants and products
    forAll(matFuel_.species(), i)
    {
        if (matFuel_.species()[i].rSign() == 1) // produced in fuel
        {
            // product
	    // stoichiometric coefficient
            scalar stoiCoeffI = rxnSpCoef_[matFuel_.species()[i].name()];

            Qrxn *= Foam::pow(xFuel_[i], stoiCoeffI);

            scalarField Hi = stoiCoeffI* // enthalpie of formation of species at temperature
            (
                 matFuel_.species()[i].hForm()
                 + matFuel_.molarCp(i, Tr_, anodeT_)
            );
            deltaHrxn += Hi;

            scalarField Si = stoiCoeffI* // entropie of formation of species at temperature
            (
                 matFuel_.species()[i].sForm()
                 + matFuel_.molarCpS(i, Tr_, anodeT_)
            );
            deltaSrxn += Si;

            if (firstTime_)
            {
                Info<< matFuel_.species()[i].name() << "  " << stoiCoeffI << nl;
                Info<< "   mean H(T) = " << gAverage(Hi) << nl;
                Info<< "   mean S(T) = " << gAverage(Si) << nl;
            }
        }
        else if (matFuel_.species()[i].rSign() == -1) // consumed fuel
        {
            // reactant
            scalar stoiCoeffI = rxnSpCoef_[matFuel_.species()[i].name()];

            Qrxn /= Foam::pow(xFuel_[i], stoiCoeffI);

            scalarField Hi = stoiCoeffI*
            (
                 matFuel_.species()[i].hForm()
                 + matFuel_.molarCp(i, Tr_, anodeT_)
            );
            deltaHrxn -= Hi;

            scalarField Si = stoiCoeffI*
            (
                 matFuel_.species()[i].sForm()
                 + matFuel_.molarCpS(i, Tr_, anodeT_)
            );
            deltaSrxn -= Si;
            if (firstTime_)
            {
                Info<< matFuel_.species()[i].name() << "  " << stoiCoeffI << nl;
                Info<< "   mean H(T) = " << gAverage(Hi) << nl;
                Info<< "   mean S(T) = " << gAverage(Si) << nl;
            }
        }
    }


    // update with air reactants and products
    forAll(matAir_.species(), i)
    {
        if (matAir_.species()[i].rSign() == 1)
        {
            // product
            scalar stoiCoeffI = rxnSpCoef_[matAir_.species()[i].name()];

            // do not add vapour to Nernst if low temperature PEM
            if (matAir_.species()[i].name() != "H2O")
            {
                Qrxn *= Foam::pow(xAir_[i], stoiCoeffI);
            }
            scalarField Hi = stoiCoeffI*
            (
                 matAir_.species()[i].hForm()
                 + matAir_.molarCp(i, Tr_, anodeT_)
            );
            deltaHrxn += Hi;

            scalarField Si = stoiCoeffI*
            (
                matAir_.species()[i].sForm()
               + matAir_.molarCpS(i, Tr_, anodeT_)
            );
            deltaSrxn += Si;

            if (firstTime_)
            {
                Info<< matAir_.species()[i].name() << "  " << stoiCoeffI << nl;
                Info<< "   mean H(T) = " << gAverage(Hi) << nl;
                Info<< "   mean S(T) = " << gAverage(Si) << nl;
            }
        }
        else if (matAir_.species()[i].rSign() == -1)
        {
            scalar stoiCoeffI = rxnSpCoef_[matAir_.species()[i].name()];

            Qrxn /= Foam::pow(xAir_[i], stoiCoeffI);

            scalarField Hi = stoiCoeffI*
            (
                 matAir_.species()[i].hForm()
                 + matAir_.molarCp(i, Tr_, anodeT_)
            );
            deltaHrxn -= Hi;

            scalarField Si = stoiCoeffI*
            (
                 matAir_.species()[i].sForm()
                 + matAir_.molarCpS(i, Tr_, anodeT_)
            );
            deltaSrxn -= Si;

            if (firstTime_)
            {
                Info<< matAir_.species()[i].name() << "  " << stoiCoeffI << nl;
                Info<< "   mean H(T) = " << gAverage(Hi) << nl;
                Info<< "   mean S(T) = " << gAverage(Si) << nl;
            }
        }
    }

    // Gibbs
    scalarField deltaGrxn = deltaHrxn - anodeT_*deltaSrxn;

    if (firstTime_)
    {
        Info<< "mean deltaHrxn = " << gAverage(deltaHrxn) << nl
            << "mean anodeT_    = " << gAverage(anodeT_)    << nl
            << "mean deltaSrxn = " << gAverage(deltaSrxn) << nl
            << "mean TdeltaS   = " << gAverage(anodeT_*deltaSrxn) << nl
            << "mean deltaG    = " << gAverage(deltaGrxn) << nl
        << endl;
    }

    // sum of Nernst voltages
    Nernst_ =
        (-deltaGrxn - Rgas_*anodeT_*Foam::log(Qrxn))/(rxnSpCoef_["e"]*F_);

    check("Nernst", Nernst_, 0, 1.5);

    scalarField E0 = -deltaGrxn/(rxnSpCoef_["e"]*F_);
    check("E0", E0, -GREAT, GREAT);

    scalarField activity = (Rgas_*anodeT_*Foam::log(Qrxn))/(rxnSpCoef_["e"]*F_);
    check("activity", activity, -GREAT, GREAT);

    scalarField etaConcentration = 
	(Rgas_*anodeT_*Foam::log(Qrxn))/(rxnSpCoef_["e"]*F_);
    check("etaConcentration", etaConcentration, -GREAT, GREAT);
}

void electrochemistryModel::calcElectrochemicalHeating()
{
    // calculate volumetric electrochemical heating
    Info<< nl << "Calculating electrochemical heating" << endl;

    // iEA applied in cell volumes (not patch)
    scalarField iEA = idensity_.primitiveFieldRef();

    // Following Cp(T) are 6th degree (7th order) polynomials in T[K]/1000, from
    // Todd & Young, Thermodynamic and transport properties of gases for use in
    // solid oxide fuel cell modelling, J. Power Sources 110 (2002) pp186-200.
    // Cp(T) = sum(a[k]*(T/1000)^k), k = 0,..,6
    // Enthalpies are definite integrals from reference to ambient temperature.

    scalar hFormSum = 0; // accumulator for enthalpy of formation
                         // normalized by n electrons assoc'd with ionization

    // air side accumulators
    // for simplicity, all is defined at anode patch with anode patch temperature used
    scalarField hReactSrcAir(anodeT_.size(),0);
    scalarField hProdSrcAir(anodeT_.size(),0);

    PtrList<scalarField> hSpAir(matAir_.species().size());
    forAll(matAir_.species(), i)
    {
        if(matAir_.species()[i].ne() != 0)
        {
            polyToddYoung& CpI = matAir_.molarCpModel()[i];        // (J/mol)
            scalarField hSpI = CpI.polyInt(Tr_, anodeT_); // Tr = reference T.
            const refPtr<scalarField>& phSpI = hSpI;
            hSpAir.set(i, phSpI); // write molar cp in "hSmatAir_.p()"

            // sum of the enthalpies of formation, as written in one of the dicts
            hFormSum += matAir_.species()[i].hForm()/matAir_.species()[i].ne();

            if(matAir_.species()[i].rSign() > 0)
            {
                // cp * delta T -> accounts probably for the different working 
		// temperature compared to the reference temperature of the 
		// standard enthalpy of formation
                hProdSrcAir += hSpI/matAir_.species()[i].ne();
            }
            else if(matAir_.species()[i].rSign() < 0)
            {
                // reactant
                hReactSrcAir += hSpI/matAir_.species()[i].ne();
            }
            else
            {
                Info<< "ERROR: rSign = 0 but ne != 0" << nl;
            }
        }
    }

    // fuel side accumulators
    scalarField hReactSrcFuel(anodeT_.size(),0);
    scalarField hProdSrcFuel(anodeT_.size(),0);

    // species enthalpies
    PtrList<scalarField> hSpFuel(matFuel_.species().size());
    forAll(matFuel_.species(), i)
    {
        if(matFuel_.species()[i].ne() != 0)
        {
            polyToddYoung& CpI = matFuel_.molarCpModel()[i];          // (J/mol)
            scalarField hSpI = CpI.polyInt(Tr_, anodeT_);
	    const refPtr<scalarField>& phSpI = hSpI;
            hSpFuel.set(i, phSpI);

            hFormSum += matFuel_.species()[i].hForm()/matFuel_.species()[i].ne();

            if(matFuel_.species()[i].rSign() > 0) // produced
            {
                // enthalpy of formation
                // divide by number of electrons to get enthalpy per electron
		// ... n in final formula: Q = j/h (Delta H/(nF) - V)
                hProdSrcFuel += hSpI/matFuel_.species()[i].ne();
            }
            else if(matFuel_.species()[i].rSign() < 0)
            {
                // reactant
                hReactSrcFuel += hSpI/matFuel_.species()[i].ne();
            }
            else
            {
                Info<< "ERROR: rSign = 0 but ne != 0" << nl;
            }
        }
    }
    // current density / F / electrolyte thickness
    scalarField volMolRate = iEA/F_/ohm_.hE().value();    // (mol/(m^3 s))

    // sum of all enthalpies - converted to energy source term
    // prod=product; react=reactand
    scalarField hSource =
    (hProdSrcFuel - hReactSrcFuel + hProdSrcAir - hReactSrcAir)*volMolRate;

    // hFormSum: sum of ALL enthalpies where ne() != 0: read from file
    // hSource: difference of enthalpy of products and educts: 
    // enthalpies for T0 to Tworking
    scalarField& electrochemicalHeatingIn = 
	electrochemicalHeating_.primitiveFieldRef();
    electrochemicalHeatingIn =
    (
        -hFormSum*volMolRate - hSource - iEA*voltage_/ohm_.hE().value()
    );
    electrochemicalHeating_.correctBoundaryConditions();
    check(electrochemicalHeating_, -GREAT, GREAT, false);
}


void Foam::electrochemistryModel::calcActivationOverpotential()
{
    const scalarField& pFuelPatch = matFuel_.p().boundaryField()[pm_.anodeID()];
    // defined on fuel-anodePatch
    tmp<scalarField> etaAnode(fuelActivationOverpotential_.overPotential
    (
        anodeT_,
        pFuelPatch,
        xFuel_,
	// accounts for water blocking of reaction sites
        pow(1.-matFuel_.s().boundaryField()[pm_.anodeID()], activationFactor_),
        i_
    ));
    check("etaAnode", etaAnode(), 0, GREAT);

    const scalarField& pAirPatch = matAir_.p().boundaryField()[pm_.cathodeID()];
    // compute eta cathode on air-cathode patch
    tmp<scalarField> etaCathode(airActivationOverpotential_.overPotential
    (
        cathodeT_,
        pAirPatch,
        xAir_, // note: i is defined at anode patch
        pow(1.-matAir_.s().boundaryField()[pm_.cathodeID()], activationFactor_),
        pm_.anodeToCathode().faceInterpolate(i_)
    ));
    check("etaCathode", etaCathode(), 0, GREAT);

    // Save data for output
    // eta is defined on air/fuel patch, but eta on electrolyte

    const labelUList& aOwner = 
	electrolyteMesh_.boundary()[pm_.electrolyteAnodeID()].faceCells();
    scalarField etaAnodeElectrolyte = 
	pm_.fuelAnodeToElectrolyteAnode().faceInterpolate(etaAnode());

    forAll(etaAnodeElectrolyte, facei)
    {
        etaA_[aOwner[facei]] = etaAnodeElectrolyte[facei];
    }
    etaA_.correctBoundaryConditions();
    etaA_.boundaryFieldRef()[pm_.electrolyteAnodeID()] == etaAnode();

    const labelUList& cOwner = 
	electrolyteMesh_.boundary()[pm_.electrolyteCathodeID()].faceCells();
    scalarField etaCathodeElectrolyte = 
	pm_.airCathodeToElectrolyteCathode().faceInterpolate(etaCathode());

    forAll(etaCathodeElectrolyte,facei)
    {
        etaC_[cOwner[facei]] = etaCathode()[facei];
    }
    etaC_.correctBoundaryConditions();
    etaC_.boundaryFieldRef()[pm_.electrolyteCathodeID()] == etaCathode();
}


void Foam::electrochemistryModel::getTemperature()
{
    // Interpolate global temperature to electrolyte/fuel interface
    // ------------------------------------------------------------
    surfaceScalarField Ts = fvc::interpolate(Tcell_);

    forAll (anodeT_, faceI)
    {
        anodeT_[faceI] = Ts[map_.electrolyteAnodeMap()[faceI]];
    }
    check("anodeT", anodeT_, Tr_ - 50., 2*Tr_);

    forAll (cathodeT_, faceI)
    {
        cathodeT_[faceI] = Ts[map_.electrolyteCathodeMap()[faceI]];
    }
    check("cathodeT", cathodeT_, Tr_ - 50., 2*Tr_);
}


void Foam::electrochemistryModel::calcMoleFractions()
{
    // Interpolate cathode mole fractions of oxidant species to anode
    // --------------------------------------------------------------
    scalarField sumAYjOnMj    //sum(Yj/MWj)
    (
        Yair_[0].boundaryField()[pm_.cathodeID()].size(),
        0
    );

    forAll(Yair_, j)
    {
        sumAYjOnMj += 
	    Yair_[j].boundaryField()[pm_.cathodeID()]/matAir_.species()[j].MW();
    }

    forAll(matAir_.species(), s)
    {
	xAir_[s] *= 0;
        if (matAir_.species()[s].rSign() != 0)    // reactant or product
        {
            xAir_[s] =
                pm_.cathodeToAnode().faceInterpolate
                (
                    Yair_[s].boundaryField()[pm_.cathodeID()]/matAir_.species()[s].MW()
                    /sumAYjOnMj
                );

            // ensure positivity
            xAir_[s] = Foam::max(xAir_[s], Foam::doubleScalarSMALL);

            check(matAir_.species()[s].name(),xAir_[s], 0., 1.);
        }
    }
    // Calculate anode mole fractions of fuel reactant and product species
    // -------------------------------------------------------------------
    scalarField sumFYjOnMj    //sum(Yj/MWj)
    (
        Yfuel_[0].boundaryField()[pm_.anodeID()].size(),
        0
    );

    forAll(Yfuel_, j)
    {
        sumFYjOnMj += Yfuel_[j].boundaryField()[pm_.anodeID()]/matFuel_.species()[j].MW();
    }

    forAll(matFuel_.species(),s)
    {
	xFuel_[s] *= 0;

        if(matFuel_.species()[s].rSign() != 0)    // reactant or product
        {

            xFuel_[s] =
            (
                Yfuel_[s].boundaryField()[pm_.anodeID()]/matFuel_.species()[s].MW()
                /sumFYjOnMj
            );

            // ensure positivity
            xFuel_[s] = Foam::max(xFuel_[s], Foam::doubleScalarSMALL);

            check(matFuel_.species()[s].name(), xFuel_[s], 0., 1.);
        }
    }
}


void Foam::electrochemistryModel::correctVoltage()
{
    label znId1 = electrolyteMesh_.cellZones().findZoneID("layer");
    Info<< nl << "Solve current" << endl;

    // number of layers in electrolyte
    for (int counter1=1; counter1<=numCells_; counter1++)
    {
        Info<< nl << "layer " << counter1 << endl;

        char rLayer[16]; // string which will contain the number

	// %d makes the result be a decimal integer
        sprintf (rLayer, "layer%d", counter1 );

        znId1 = electrolyteMesh_.cellZones().findZoneID(rLayer);
        if (znId1 < 0)
	{
	    Info<< "Warning: Zone 'layerI' in electrolyte not found!" << endl;
	}

        labelList zn1(electrolyteMesh_.cellZones()[znId1]);

        dimensionedScalar ibar ("ibar", dimensionSet(0, -2, 0, 0, 0, 1, 0), 0.0);
        dimensionedScalar temp ("temp", dimensionSet(0, 0, 0, 1, 0, 0, 0), 0.0);

        ibar.value() = 0.0;         // mean current density
        scalar rCurrentDV = 0.0;    // J * V
        scalar volume = 0.0;
        scalar sumVolume = 0.0;
        scalar counter2 = 0.0;

        forAll(zn1, j)
        {
            volume = mag(electrolyteMesh_.V()[zn1[j]]);
            rCurrentDV += idensity_.primitiveField()[zn1[j]]*volume;
            sumVolume += volume;
            counter2 += 1;
	    // voltage of this layer of the electrolyte
	    // for potentiostatic run, this will be written in log file later
            V_.value() = voltage_[zn1[j]];
        }

        reduce(sumVolume,plusOp<scalar>());
        reduce(rCurrentDV,plusOp<scalar>());
        reduce(counter2,plusOp<scalar>());

        ibar.value() = rCurrentDV/sumVolume;        // mean current density

        check("Nernst", Nernst_, -GREAT, 1.5);

        // voltage_ correction
        if (galvanostatic_)
        {
            V_ += dimensionedScalar("Rhat", dimensionSet(1,4,-3,0,0,-2,0), Rhat_)
		*(ibar - ibar0_); // ibar0 is what we want!
                                  // we change V so that we reach it

            Info<< "ibar0 = " << ibar0_.value()
                << "    ibar = " << ibar.value()
                << "    V = " << V_.value() << endl;

            forAll(zn1, j)
            {
                voltage_[zn1[j]] = V_.value(); // all get the same voltage_
            }
        }
        else
        {
            Info<< "    ibar = " << ibar.value()
                << "    V = " << V_.value() << endl;
        }
    }
}


void Foam::electrochemistryModel::updateBoundaryConditions()
{
    // Calculate species sources and sinks and set
    // boundary conditions for mass fractions and velocity
    // at fluid/electrode interfaces

    // Air side
    {
        Info<< nl << "air species electrochemical fluxes and YEqn BCs" << nl;

        label nSpec = matAir_.species().size(); // number of species
        List<scalarField> mflux(nSpec); // mass flux by species
        List<Switch> isFlux(nSpec, false); // true if species contributes
        scalarField mfluxSum(i_.size(), 0); // sum over species

        // mass fluxes for electrochemically active species
        forAll(matAir_.species(), s)
        {
            if(matAir_.species()[s].ne() != 0)
            {
                mflux[s] = // mass flux of single species
                (
		    // rSign: consumed/produced/neutral; MW: molecular weight kg/kmol
                    matAir_.species()[s].rSign()*matAir_.species()[s].MW()
		    // FF: Faraday (kg/kmol); ne: number of electrons; i: current density
                    *i_/(FF_*matAir_.species()[s].ne())
                );
                mfluxSum += mflux[s];
                isFlux[s] = true;
            }
            Info << "s species[] isFlux[] mflux[] = "
                 << s << " " << matAir_.species()[s].name() << " " << isFlux[s] << " "
                 << Foam::gSum(mflux[s]) << nl;
        }
        Info << endl;

        // add mass flux due to water transport through membrane
        // water is coming from fuel, and going to air, i.e. produced in air, i.e. positive
        // mFlux is ALLWAYS defined at fuel_to_electrolyte i.e. anode patch
        scalarField airWater = 
	    pm_.electrolyteAnodeToFuelAnode().faceInterpolate(relaxMem_
	    *elec_.Q().boundaryField()[pm_.electrolyteAnodeID()] 
	    + (1. - relaxMem_)
	    *elec_.Qold().boundaryField()[pm_.electrolyteAnodeID()]);

        if(mflux[matAir_.waterIndex()].size() > 0)
        {
            mflux[matAir_.waterIndex()] += airWater;
        }
        else
        {
            mflux[matAir_.waterIndex()] = airWater;
        }
        mfluxSum += airWater;
        isFlux[matAir_.waterIndex()] = true;

        Info << "sum of water flux and mean water flux into air is "
             << Foam::gSum(airWater) << " "
             << Foam::gAverage(airWater) << endl;

        // mass fraction BCs for all species except the "inert" specie
        // (inert specie mass fraction is 1-complement of all others)
        forAll(Yair_, s)
        {
            if(matAir_.species()[s].name() != matAir_.inertSpecie())
            {
                scalarField gammaCathodeS = // diffusivity * density
                (
		    // diffusivity of species
                    matAir_.diffusivity()[s].boundaryField()[pm_.cathodeID()]
                    *matAir_.rho().boundaryField()[pm_.cathodeID()]
                );

                // interface patchField
                volScalarField& Ys = Yair_[s];
                fixedGradientFvPatchScalarField& YsBC =
                    refCast<fixedGradientFvPatchScalarField>
                    (
                        Ys.boundaryFieldRef()[pm_.cathodeID()]
                    );

                // gradient boundary condition
                // note: air mass fluxes defined on anode patch 
		// (because current density is defined there)
                // -> we interpolate to cathode patch

                // initialize gradient
                YsBC.gradient() =
                    pm_.anodeToCathode().faceInterpolate(mflux[s])*(1.0 - YsBC);

                // add changes due to other species
                forAll(Yair_, t)
                {
                    if((t != s) && isFlux[t])
                    {
                        YsBC.gradient() -=
                        pm_.anodeToCathode().faceInterpolate(mflux[t])*YsBC;
                    }
                }
                YsBC.gradient() /= gammaCathodeS;
            }
        }
        Info<< endl;

        // Set the interface velocity condition
        Uair_.boundaryFieldRef()[pm_.cathodeID()] ==
        (
            -pm_.anodeToCathode().faceInterpolate(mfluxSum)
            /matAir_.rho().boundaryField()[pm_.cathodeID()]
            *(airMesh_.Sf().boundaryField()[pm_.cathodeID()])
            /(airMesh_.magSf().boundaryField()[pm_.cathodeID()])
        );

        airBalance_.calcFluxes(mflux);

    }

    // Fuel side
    {
        Info<< "fuel species electrochemical fluxes and YEqn BCs" << nl;

        label nSpec = matFuel_.species().size();
        List<scalarField> mflux(nSpec);
        List<Switch> isFlux(nSpec, false);
        scalarField mfluxSum(i_.size(), 0);

        // mass fluxes for electrochemically active species
        forAll(matFuel_.species(), s)
        {
            if(matFuel_.species()[s].ne() != 0)
            {
                mflux[s] = // consumed = negativ; produced = positive
                (
                    matFuel_.species()[s].rSign()*matFuel_.species()[s].MW()
                    *i_/(FF_*matFuel_.species()[s].ne())
                );
                mfluxSum += mflux[s];
                isFlux[s] = true;
		// consumed i.e. H2
                if(matFuel_.species()[s].rSign() < 0){mfluxOld_ = mflux[s];} 
            }
            Info<< "s species[] isFlux[] mflux[] = "
                << s << " " << matFuel_.species()[s].name() << " " << isFlux[s] << " "
                << Foam::gSum(mflux[s]) << nl;
        }
        Info<< endl;

        // add mass flux due to water transport through membrane
        // water is going from fuel to air, i.e. is leaving fuel, i.e. negative
        scalarField fuelWater = 
	    - pm_.electrolyteAnodeToFuelAnode().faceInterpolate(relaxMem_
	    *elec_.Q().boundaryField()[pm_.electrolyteAnodeID()] 
	    + (1. - relaxMem_)
	    *elec_.Qold().boundaryField()[pm_.electrolyteAnodeID()]);

        if(mflux[matFuel_.waterIndex()].size() > 0)
        {
            mflux[matFuel_.waterIndex()] += fuelWater;
        }
        else
        {
            mflux[matFuel_.waterIndex()] = fuelWater;
            Info << "water empty in fuel" << endl;
        }
        mfluxSum += fuelWater;
        isFlux[matFuel_.waterIndex()] = true;

        Info << "sum of water flux and mean water flux into fuel is "
             << Foam::gSum(fuelWater) << " "
             << Foam::gAverage(fuelWater) << endl;

        // mass fraction BCs for all species except the "inert" specie
        // (inert specie mass fraction is 1-complement of all others)
        forAll(Yfuel_, s)
        {
            if(matFuel_.species()[s].name() != matFuel_.inertSpecie())
            {

                scalarField gammaAnodeS =
                (
                    matFuel_.diffusivity()[s].boundaryField()[pm_.anodeID()]
                    *matFuel_.rho().boundaryField()[pm_.anodeID()]
                );

                // interface patchField
                volScalarField& Ys = Yfuel_[s];
                fixedGradientFvPatchScalarField& YsBC =
                    refCast<fixedGradientFvPatchScalarField>
                    (
                        Ys.boundaryFieldRef()[pm_.anodeID()]
                    );

                // gradient boundary condition
                // initialize gradient
                YsBC.gradient() = mflux[s]*(1.0 - YsBC);

                // add changes due to other species
                forAll(Yfuel_, t)
                {
                    if((t != s) && isFlux[t])
                    {
                        YsBC.gradient() -= YsBC*mflux[t];
                    }
                }

                YsBC.gradient() /= gammaAnodeS;
            }
        }
        Info << endl;

        // Set the interface velocity condition
        Ufuel_.boundaryFieldRef()[pm_.anodeID()] ==
        (
            -(mfluxSum)
            /matFuel_.rho().boundaryField()[pm_.anodeID()]
            *(fuelMesh_.Sf().boundaryField()[pm_.anodeID()])
            /(fuelMesh_.magSf().boundaryField()[pm_.anodeID()])
        );

        fuelBalance_.calcFluxes(mflux);
    }
}

// ************************************************************************* //
