/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::electrolyteModel

Description
    Computes water content in, and water transfer through membrane, and 
    provies membrane properties.

\*---------------------------------------------------------------------------*/

#ifndef electrolyteModel_H
#define electrolyteModel_H

#include "IOdictionary.H"
#include "materialDatabase.H"
#include "patchDatabase.H"
#include "checkFieldLimits.H"
#include "physicalConstants.H"
#include "ohmicOverpotentialModel.H"

namespace Foam
{

/*---------------------------------------------------------------------------*\
                    Class electrolyteModel Declaration
\*---------------------------------------------------------------------------*/

class electrolyteModel 
:
    private checkFieldLimits,
    public IOdictionary
{

private:

    // Private data

	//- Reference to electrolyte mesh
	const fvMesh& mesh_;

	//- Density
	const dimensionedScalar rho_;
	//- Heat capacity
	const dimensionedScalar cp_;
	//- Thermal conductivity
	const dimensionedScalar k_;
	//- Equivalent weight
	const dimensionedScalar Meq_;
	//- For conversion from K to °C
	const dimensionedScalar T0_;
	//- Correction of water flux through membrane
	const scalar corr_;

	//- Electrolyte thickness
	const dimensionedScalar& hE_;
	//- Properties of air
	materialDatabase& matAir_;
	//- Properties of fuel
	materialDatabase& matFuel_;
	//- Patch names and IDs
	patchDatabase& pm_;

	//- Water content of membrane
        volScalarField lambda_;
	//- Water flux through membrane
        volScalarField Q_;
	//- Water flux of last time step
        volScalarField Qold_;
	//- Diffusion coefficient
        volScalarField D_;
	//- Gradient of water flux
        volScalarField gradLambda_;
	//- Temperature
	volScalarField T_;

    // Private member functions

        //- Disallow copy construct
        electrolyteModel(const electrolyteModel&);

        //- Disallow default bitwise assignment
        void operator=(const electrolyteModel&);
	
public:

    // Constructors

        electrolyteModel
	(
	    const fvMesh&, 
	    materialDatabase&, 
	    materialDatabase&, 
	    patchDatabase&, 
	    const ohmicOverpotentialModel&
	);

    // Destructor

        ~electrolyteModel(){};


    // Member Functions

	//- Solves the water content of the membrane
	void solveWaterContent();
	//- Computes the water flux through membrane
	void solveWaterFlux(const volScalarField&);
	//- Solves water content and flux
        void solve(const volScalarField& idensity)
	{
	    solveWaterContent();
	    solveWaterFlux(idensity);
  	}
	//- Return heat capacity
	const dimensionedScalar& cp()
	{
	    return cp_;
	}
	//- Return thermal conductivity
        const dimensionedScalar& k()
	{
	    return k_;
	}
	//- Return density
        const dimensionedScalar& rho()
	{
	    return rho_;
	}
	//- Return water content of membrane
	const volScalarField& lambda()
	{
	    return lambda_;
	}
	//- Return temperature of membrane
        volScalarField& T()
	{
	    return T_;
	}
	//- Return water flux through membrane
        const volScalarField& Q()
	{
	    return Q_;
	}
	//- Return water flux of last time step
        const volScalarField& Qold()
	{
	    return Qold_;
	}
	//- Return reference to electrolyte mesh
	const fvMesh& mesh()
	{
	    return mesh_;
	}
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
