/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::flowBalance

Description
    Computes Reynolds numbers and gas/species fluxes at different patches
    to check continuity.

\*---------------------------------------------------------------------------*/

#include "flowBalance.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::flowBalance::flowBalance
(
    materialDatabase& mat,
    patchDatabase& pm,
    liquidWaterModel& water,
    phaseChange& phaseChange
)
:
    IOdictionary
    (
        IOobject
        (
            "phaseProperties",
            mat.U().mesh().time().constant(),
            mat.U().mesh(),
            IOobject::MUST_READ,
            IOobject::NO_WRITE
        )
    ),
    dHyd_("dHyd", dimLength, *this),
    Re_(dHyd_.value() != 0), // do not calc Re, if dHyd was zero
    mesh_(mat.U().mesh()),
    U_(mat.U()),
    Svl_(phaseChange.Svl()),
    rhoW_(water.rho()),
    mat_(mat),
    pm_(pm),
    inletID_(-1),
    outletID_(-1),
    reactID_(-1),
    isAir_(false)
{
    Info<< nl << "Creating mass flux model for " << mesh_.name()
	<<  "    dHyd." << mesh_.name() << "  = " << dHyd_  << endl;

    if(!Re_)
    {
        Info<< "    Hydraulic diameter = 0: not calculating Re for phase " << mesh_.name() << nl << endl;
    }

    if(mesh_.name() == pm_.airName())
    {
	inletID_ = pm_.airInletID();
        outletID_ = pm_.airOutletID();
        reactID_ = pm_.cathodeID();
	isAir_ = true;
    }
    else if(mesh_.name() == pm_.fuelName())
    {
        inletID_ = pm_.fuelInletID();
        outletID_ = pm_.fuelOutletID();
        reactID_ = pm_.anodeID();
	isAir_ = false;
    }
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::flowBalance::calcRe()
{
    if(Re_)
    {
        scalar Re = dHyd_.value()*Foam::mag
        (
            Foam::gAverage
            (
                mat_.rho().boundaryField()[inletID_]
                *U_.boundaryField()[inletID_]
                /mat_.mu().boundaryField()[inletID_]
            )
        );

        Info<< "Re:" << mesh_.name() << "  = " << Re << endl;
    }
}


void Foam::flowBalance::calcFluxes(List<scalarField>& mflux)
{
    // total gas mass rate at inlet
    scalarField rateInlet =
    (
        mat_.rho().boundaryField()[inletID_]
        *(
             U_.boundaryField()[inletID_]
             &
             mesh_.Sf().boundaryField()[inletID_]
         )
	*mat_.epsilon0().boundaryField()[inletID_]
	*(1.0 - mat_.s().boundaryField()[inletID_])
    );

    // total gas mass rate at outlet
    scalarField rateOutlet =
    (
        mat_.rho().boundaryField()[outletID_]
        *(
             U_.boundaryField()[outletID_]
             &
             mesh_.Sf().boundaryField()[outletID_]
         )
        *mat_.epsilon0().boundaryField()[outletID_]
        *(1.0 - mat_.s().boundaryField()[outletID_])
    );

    // total gas mass rate at electrolyte interface
    scalarField rateReaction =
    (
        mat_.rho().boundaryField()[reactID_]
        *(
             U_.boundaryField()[reactID_]
             &
             mesh_.Sf().boundaryField()[reactID_]
         )
        *mat_.epsilon0().boundaryField()[reactID_]
        *(1.0 - mat_.s().boundaryField()[reactID_])
    );

    // total phase change rate (Svl in kg/m3/s)
    scalarField phaseChange =
    (
        Svl_*mesh_.V()
    );

    // total liquid water mass rate at inlet
    scalarField srateInlet =
    (
        rhoW_.value()*mat_.s().boundaryField()[inletID_]
        *(
             U_.boundaryField()[inletID_]
             &
             mesh_.Sf().boundaryField()[inletID_]
         )
        *mat_.epsilon0().boundaryField()[inletID_]
    );

    // total liquid water mass rate at outlet
    scalarField srateOutlet =
    (
        rhoW_.value()*mat_.s().boundaryField()[outletID_]
        *(
             U_.boundaryField()[outletID_]
             &
             mesh_.Sf().boundaryField()[outletID_]
         )
        *mat_.epsilon0().boundaryField()[outletID_]
    );

    Info<< mesh_.name() << " mass rates " << nl
        << "    gas inlet: " << Foam::mag(Foam::gSum(rateInlet))
        << ", gas outlet: " << Foam::mag(Foam::gSum(rateOutlet))
        << ", gas cathode: " << Foam::mag(Foam::gSum(rateReaction))
        << ", water inlet: " << Foam::mag(Foam::gSum(srateInlet))
        << ", water outlet: " << Foam::mag(Foam::gSum(srateOutlet))
        << ", water phase change: " << Foam::mag(Foam::gSum(phaseChange))
        << nl << endl;

    // --------------------------------------------------------------------

    forAll(mat_.species(), I)
    {
        scalar rateInletI = gSum
        (
            rateInlet*mat_.Y()[I].boundaryField()[inletID_]
        );

        scalar rateOutletI = gSum
        (
            rateOutlet*mat_.Y()[I].boundaryField()[outletID_]
        );

	scalar rateElecI = 0;

        if(mat_.species()[I].rSign() < 0)
	{
	    if(isAir_)
	    {
            	rateElecI = gSum
            	(
                    pm_.anodeToCathode().faceInterpolate(mflux[I])
                    *mesh_.magSf().boundaryField()[reactID_]
                );
	    }
	    else
	    {
		rateElecI = gSum
		(
                    mflux[I]*mesh_.magSf().boundaryField()[reactID_]
            	);
	    }    
	}

        Info<< "    " << mat_.species()[I].name() << " mass rates " << nl; 
        Info<< "        inlet = " << Foam::mag(rateInletI)
            << ", outlet = " << Foam::mag(rateOutletI)
	    << ", chem = " << Foam::mag(rateElecI)
	    << nl;


        Info<< "    " << mat_.species()[I].name() 
	    << " utilization by chemRate = "
            << Foam::mag(rateElecI/rateInletI)
            << nl
            << "        utilization by outRate  = "
            << 1.0 - Foam::mag(rateOutletI/rateInletI)
            << nl;

        Info<< endl;

        if(mat_.species()[I].name() == "H2O")
        {
            Info<< "    Water balance: " << nl
                << "        gas inlet: " << rateInletI
                << ", gas outlet: " << rateOutletI
                << ", chem: " << rateElecI
                << ", phase change: " << Foam::mag(Foam::gSum(phaseChange))
                << ", liquid inlet: " << Foam::mag(Foam::gSum(srateInlet))
                << ", liquid outlet: " << Foam::mag(Foam::gSum(srateOutlet))
                << nl << endl;
        }
    }
}

// ************************************************************************* //
