/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::mapMesh

Description
    Provides an interpolation between child and parent mesh.

\*---------------------------------------------------------------------------*/

#include "mapMesh.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::mapMesh::mapMesh
(
    const fvMesh& mesh,
    patchDatabase& pm
)
:
    mesh_(mesh),
    pm_(pm),
    faceRegionAddressing_
    (
        IOobject
        (
            "faceRegionAddressing",
            mesh.time().findInstance(mesh.meshDir(), "faces"),
            polyMesh::meshSubDir,
            mesh,
            IOobject::MUST_READ
        )
    ),
    faceMap_(faceRegionAddressing_.size()),
    faceMask_(faceRegionAddressing_.size()),
    cellMap_
    (
        IOobject
        (
            "cellRegionAddressing",
            mesh.time().findInstance(mesh.meshDir(), "faces"),
            polyMesh::meshSubDir,
            mesh,
            IOobject::MUST_READ
        )
    ),
    patchesMap_
    (
        IOobject
        (
            "boundaryRegionAddressing",
            mesh.time().findInstance(mesh.meshDir(), "faces"),
            polyMesh::meshSubDir,
            mesh,
            IOobject::MUST_READ
        )
    )
{
    // faceMask: same sign - same orientation of faces
    // faceMask: different sign - opposite orientation of faces
    // as "0" cannot have a sign, numbering starts with "1"
    // therefore, 1 needs to be subtracted from faceRegionA. to get the mapping

    forAll(faceMap_, i)
    {
        faceMap_[i] = mag(faceRegionAddressing_[i]) - 1;
        faceMask_[i] = sign(faceRegionAddressing_[i]);
    }
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

void Foam::mapMesh::rmap
(
    volScalarField& targetField,
    const volScalarField& sourceField
)
{
    scalarField& targetFieldIn = targetField.primitiveFieldRef();
    targetFieldIn.rmap(sourceField, cellMap_);
}


void Foam::mapMesh::rmap
(
    volScalarField& targetField,
    const dimensionedScalar& sourceValue
)
{
    scalarField& targetFieldIn = targetField.primitiveFieldRef();
    scalarField sourceField(mesh_.nCells(), sourceValue.value());

    targetFieldIn.rmap(sourceField, cellMap_);
}


void Foam::mapMesh::rmap
(
    volScalarField& targetField,
    scalarField& sourceField
)
{
    scalarField& targetFieldIn = targetField.primitiveFieldRef();
    targetFieldIn.rmap(sourceField, cellMap_);
}


void Foam::mapMesh::rmap
(
    surfaceScalarField& targetField,
    const surfaceScalarField& sourceField
)
{
    // internal faces
    labelList internalFaceMap
    (
        SubList<label>(faceMap_, mesh_.nInternalFaces())
    );

    scalarField internalFaceMask
    (
        scalarField::subField(faceMask_, mesh_.nInternalFaces())
    );

    // map internal faces first
    // mask is 1 or -1 ... depending whether faces have same direction or not
    scalarField& targetFieldIn = targetField.primitiveFieldRef();
    targetFieldIn.rmap(sourceField.primitiveField()*internalFaceMask, internalFaceMap);

    // map the patches
    forAll (patchesMap_, patchI)
    {
        if
        (
            patchesMap_[patchI] > -1 // patch does not exist in parent mesh (e.g. _to_ patch)
         && patchesMap_[patchI] < targetField.mesh().boundary().size() // processor patches
        )
        {
            // current field patch mapper
            labelField curFpm
            (
                labelField::subField
                (
                    faceMap_,
                    mesh_.boundary()[patchI].size(),
                    mesh_.boundary()[patchI].patch().start()
                )
            );

            scalarField curMask
            (
                scalarField::subField
                (
                    faceMask_,
                    mesh_.boundary()[patchI].size(),
                    mesh_.boundary()[patchI].patch().start()
                )
            );

            curFpm -= targetField.mesh().boundary()
                [patchesMap_[patchI]].patch().start();

            targetField.boundaryFieldRef()[patchesMap_[patchI]].
                scalarField::rmap
                (
                    sourceField.boundaryField()[patchI]*curMask,
                    curFpm
                );
	}
    }

    // for air and fuel, map reacting patches, as well
    if(mesh_.name() == pm_.airName() || mesh_.name() == pm_.fuelName())
    {
        labelField reactingPatchMap
        (
            labelField::subField
            (
                faceMap_,
                pm_.reactingPatch(mesh_).size(),
                pm_.reactingPatch(mesh_).start()
            )
        );

        scalarField reactingPatchMask
        (
            scalarField::subField
            (
                faceMask_,
                mesh_.boundary()[pm_.reactingPatchID(mesh_)].size(),
                mesh_.boundary()[pm_.reactingPatchID(mesh_)].patch().start()
            )
        );

        scalarField reactingPatchField = 
	    sourceField.boundaryField()[pm_.reactingPatchID(mesh_)];
        forAll(reactingPatchField, faceI)
        {
            targetField[reactingPatchMap[faceI]] = 
		reactingPatchField[faceI]*reactingPatchMask[faceI];
        }
    }
}


void Foam::mapMesh::zeroInternalFlux
(
    surfaceScalarField& field
)
{
    scalarField& fieldIn = field.primitiveFieldRef();

    labelList internalFaceMap
    (
        SubList<label>
        (
            faceMap_,
            mesh_.nInternalFaces()
        )
    );

    scalarField zf(internalFaceMap.size(), 0);

    // field is on the parent mesh
    // set only those to zero, which belong to the child mesh
    fieldIn.rmap
    (
        zf,
        internalFaceMap
    );
}

// ************************************************************************* //
