/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::patchDatabase

Description
    Collects all patch names and their IDs, and provides patch to patch
    interpolation for different child meshes.

\*---------------------------------------------------------------------------*/

#ifndef patchDatabase_H
#define patchDatabase_H

#include "fvMesh.H"
#include "Time.H"
#include "IOdictionary.H"
#include "patchToPatchInterpolation.H"
#include "checkFieldLimits.H"

namespace Foam
{

/*---------------------------------------------------------------------------*\
                     Class patchDatabase Declaration
\*---------------------------------------------------------------------------*/

class patchDatabase
:
    private checkFieldLimits
{
private:

    // Private data

	//- Reference to global mesh
        const fvMesh& mesh_;
	//- Reference to air mesh
        const fvMesh& airMesh_;
	//- Reference to fuel mesh
        const fvMesh& fuelMesh_;
	//- Reference to electrolyte mesh
        const fvMesh& electrolyteMesh_;

	//- Dictionary for reading patch names
        const IOdictionary cellProperties_;

    	//- Name of anode patch
        const word anodeName_;
	//- Name of fuel inlet patch
        const word fuelInletName_;
	//- Name of fuel outlet patch
        const word fuelOutletName_;

	//- Name of cathode patch
        const word cathodeName_;
	//- Name of air inlet patch
        const word airInletName_;
	//- Name of air outlet patch
        const word airOutletName_;

	//- Name of electrolyte anode
        const word electrolyteAnodeName_;
	//- Name of electrolyte cathode
        const word electrolyteCathodeName_;

        const label anodeID_;
        const label fuelInletID_;
        const label fuelOutletID_;

        const label cathodeID_;
        const label airInletID_;
        const label airOutletID_;

        const label electrolyteAnodeID_;
        const label electrolyteCathodeID_;

	//- Reference to anode patch
        const polyPatch& anodePatch_;
	//- Reference to cathode patch
        const polyPatch& cathodePatch_;
	//- Reference to electrolyte anode patch
        const polyPatch& electrolyteAnodePatch_;
	//- Reference to electrolyte cathode patch
        const polyPatch& electrolyteCathodePatch_;

        //- Interpolation from anode to cathode
        patchToPatchInterpolation anodeToCathode_;
        patchToPatchInterpolation cathodeToAnode_;
        patchToPatchInterpolation fuelAnodeToElectrolyteAnode_;
        patchToPatchInterpolation airCathodeToElectrolyteCathode_;
        patchToPatchInterpolation electrolyteAnodeToFuelAnode_;


    // Private member functions

        //- Disallow copy construct
        patchDatabase(const patchDatabase&) = delete;

        //- Disallow default bitwise assignment
        void operator=(const patchDatabase&) = delete;

public:

    // Constructors

        patchDatabase(const fvMesh&, const fvMesh&, const fvMesh&, const fvMesh&);

    // Destructor

        ~patchDatabase(){};


    // Member Functions

        //- Return anode name
        const word& anodeName() const
        {
            return anodeName_;
        };

	//- Return fuel inlet name
        const word& fuelInletName() const
        {
            return fuelInletName_;
        };

	//- Return fuel outlet name
        const word& fuelOutletName() const
        {
            return fuelOutletName_;
        };

        const word& cathodeName() const
        {
            return cathodeName_;
        };

        const word& airInletName() const
        {
            return airInletName_;
        };

        const word& airOutletName() const
        {
            return airOutletName_;
        };

        const word& electrolyteAnodeName() const
        {
            return electrolyteAnodeName_;
        };

        const word& electrolyteCathodeName() const
        {
            return electrolyteCathodeName_;
        };


        //- Return ID of the anode patch
        label anodeID() const
        {
    	    return anodeID_;
        }

        label fuelInletID() const
        {
            return fuelInletID_;
        }

        label fuelOutletID() const
        {
            return fuelOutletID_;
        }

        label cathodeID() const
        {
            return cathodeID_;
        }

        label airInletID() const
        {
            return airInletID_;
        }

        label airOutletID() const
        {
            return airOutletID_;
        }

        label electrolyteAnodeID() const
        {
            return electrolyteAnodeID_;
        }

        label electrolyteCathodeID() const
        {
            return electrolyteCathodeID_;
        }

	//- Return ID of the reacting patch (anode or cathode)
        label reactingPatchID(const fvMesh& mesh) const
        {
    	    if (mesh.name() == airMesh_.name())
    	    {
	        return cathodeID();
            }
	    else if (mesh.name() == fuelMesh_.name())
            {
	        return anodeID();
            }
	    else
	    {
	        FatalError << "Mesh " << mesh.name() << " is neighter air nor fuel" << abort(FatalError);
    	    }
        }

	//- Return reference to anode patch
        const polyPatch& anodePatch() const
        {
    	    return anodePatch_;
        }

	//- Return reference to cathode patch
        const polyPatch& cathodePatch() const
        {
	    return cathodePatch_;
        }

	//- Return reference to electrolyte anode patch
        const polyPatch& electrolyteAnodePatch() const
        {
    	    return electrolyteAnodePatch_;
        }

	//- Return reference to electrolyte cathode patch
        const polyPatch& electrolyteCathodePatch() const
        {
    	    return electrolyteCathodePatch_;
        }

	//- Return reference to reacting patch (anode or cathode)
        const polyPatch& reactingPatch(const fvMesh& mesh) const
        {
            if (mesh.name() == airMesh_.name())
            {
                return cathodePatch();
            }
            else if (mesh.name() == fuelMesh_.name())
            {
                return anodePatch();
            }
            else
            {
                FatalError << "Mesh " << mesh.name() << " is neighter air nor fuel" << abort(FatalError);
            }
        }

        //- Interpoltion from anode to cathode
        patchToPatchInterpolation& anodeToCathode()
        {
            return anodeToCathode_;
        }

        patchToPatchInterpolation& cathodeToAnode()
        {
    	    return cathodeToAnode_;
        }

        patchToPatchInterpolation& fuelAnodeToElectrolyteAnode()
        {
    	    return fuelAnodeToElectrolyteAnode_;
        }

        patchToPatchInterpolation& airCathodeToElectrolyteCathode()
        {
	    return airCathodeToElectrolyteCathode_;
        }

        patchToPatchInterpolation& electrolyteAnodeToFuelAnode()
        {
    	    return electrolyteAnodeToFuelAnode_;
        }

	//- Return name of air
        const word& airName() const
        {
	    return airMesh_.name();
        }
  
	//- Return name of fuel
        const word& fuelName() const
        {
    	    return fuelMesh_.name();
        }
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
