/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::phaseChange

Description
    Computes phase change rate depending on the partial pressure.

\*---------------------------------------------------------------------------*/

#ifndef phaseChange_H
#define phaseChange_H

#include "IOdictionary.H"
#include "materialDatabase.H"
#include "checkFieldLimits.H"

namespace Foam
{

/*---------------------------------------------------------------------------*\
                      Class phaseChange Declaration
\*---------------------------------------------------------------------------*/

class phaseChange 
:
    private checkFieldLimits
{
private:

    // Private data

	//- For converting Kelvin to degr. Celsius
	const dimensionedScalar T0_;
	//- Condensation rate
        const dimensionedScalar gammaCond_;
	//- Evaporation rate
        const dimensionedScalar gammaEvap_;
	//- Maximum allowed liquid water content
	const scalar sMax_;
	//- Phase change activated or not
        bool phaseChangeOn_;

	//- Reference to temperature
        const volScalarField& T_;
	//- Gas properties
        materialDatabase& mat_;

	//- Source term vapour-liquid phase change
        volScalarField Svl_;
	//- Saturation pressure
	volScalarField pSat_;
	//- Partial pressure
	volScalarField partialP_;
	//- Humidity
	volScalarField humidity_;

    // Private member functions

        //- Disallow copy construct
        phaseChange(const phaseChange&) = delete;

        //- Disallow default bitwise assignment
        void operator=(const phaseChange&) = delete;
	
public:

    // Constructors

        phaseChange(const IOdictionary&, materialDatabase&);

    // Destructor

        ~phaseChange(){};


    // Member Functions

	//- Return saturation pressure
	const volScalarField& pSat() const
	{
	    return pSat_;
	}

	//- Return source term vapour-liquid phase change
	volScalarField& Svl()
	{
	    return Svl_;
	}

	//- Compute phase change
	void solve();
};


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

#endif

// ************************************************************************* //
