/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright held by original author
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by the
    Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM; if not, write to the Free Software Foundation,
    Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

Class
    Foam::polyToddYoung

Description
    Polynomial function for different material properties. Coefficients
    will be read from file

\*---------------------------------------------------------------------------*/

#include "polyToddYoung.H"
#include "scalar.H"
#include "scalarField.H"


// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

Foam::polyToddYoung::polyToddYoung
(
    scalar a0, scalar a1, scalar a2,
    scalar a3, scalar a4, scalar a5, scalar a6
)
:
    a0_(a0),
    a1_(a1),
    a2_(a2),
    a3_(a3),
    a4_(a4),
    a5_(a5),
    a6_(a6)
{}
 

Foam::polyToddYoung::polyToddYoung(Istream& is)
{
    coeffArray tyCoeffs_;

    forAll(tyCoeffs_, i)
    {
        is >> tyCoeffs_[i];
    }

    // Check state of Istream
    is.check("polyToddYoung::polyToddYoung(Istream& is)");

    a0_ = tyCoeffs_[0];
    a1_ = tyCoeffs_[1];
    a2_ = tyCoeffs_[2];
    a3_ = tyCoeffs_[3];
    a4_ = tyCoeffs_[4];
    a5_ = tyCoeffs_[5];
    a6_ = tyCoeffs_[6];
}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

//- Evaluate the polynomial function and return the result
Foam::scalar Foam::polyToddYoung::polyVal(scalar T)
{
    scalar t = T*1.0e-3;
    return (((((a6_*t + a5_)*t + a4_)*t + a3_)*t + a2_)*t + a1_)*t + a0_;
}


//- Evaluate the polynomial function and return the result
Foam::tmp<Foam::scalarField> Foam::polyToddYoung::polyVal(scalarField T)
{
    tmp<scalarField> pField
    (
        new scalarField (T.size(), 1.0)
    );
    scalarField& field = pField.ref();

    scalarField t = T*1.0e-3;
    field = (((((a6_*t + a5_)*t + a4_)*t + a3_)*t + a2_)*t + a1_)*t + a0_;

    return pField;
}


//- Evaluate the integral of the polynomial function and return the result
// Assumes zero for constant of integration
Foam::scalar Foam::polyToddYoung::polyInt(scalar T)
{
    scalar t = T*1.0e-3;
    return ((((((
    (
        a6_/7.0*t
        + a5_/6.0)*t
        + a4_/5.0)*t
        + a3_/4.0)*t
        + a2_/3.0)*t
        + a1_/2.0)*t
        + a0_)*t
    )*1.0e3;
}


//- Evaluate the integral of the polynomial function and return the result
//  Assumes zero for constant of integration
Foam::tmp<Foam::scalarField> Foam::polyToddYoung::polyInt(scalarField T)
{
    tmp<scalarField> pField
    (
        new scalarField (T.size(), 1.0)
    );
    scalarField& field = pField.ref();

    scalarField t = T*1.0e-3;
    field = ((((((
    (
        a6_/7.0*t
        + a5_/6.0)*t
        + a4_/5.0)*t
        + a3_/4.0)*t
        + a2_/3.0)*t
        + a1_/2.0)*t
        + a0_)*t
    )*1.0e3;

    return pField;
}


//- Evaluate the definite integral of the polynomial function 
//  from Tlow to Thigh and return the result
//  (used with Cp polynomial for enthalpy)
Foam::scalar Foam::polyToddYoung::polyInt(scalar Tlow, scalar Thigh)
{
    return(polyInt(Thigh) - polyInt(Tlow));
}


//- Evaluate the definite integral of the polynomial function 
//  from Tlow to Thigh and return the result
Foam::tmp<Foam::scalarField> 
Foam::polyToddYoung::polyInt(scalar Tlow, scalarField Thigh)
{
    scalarField TlowF(Thigh.size(), Tlow);
    return(polyInt(Thigh) - polyInt(TlowF));
}


//- Evaluate the definite integral of the polynomial function 
//  from Tlow to Thigh and return the result
Foam::tmp<Foam::scalarField> 
Foam::polyToddYoung::polyInt(scalarField Tlow, scalarField Thigh)
{
    return(polyInt(Thigh) - polyInt(Tlow));
}


//- Evaluate the integral of the polynomial function divided by T
//  and return the result.  Assumes zero for constant of integration
//  (used with Cp polynomial for entropy)
Foam::scalar Foam::polyToddYoung::polyIntS(scalar T)
{
    scalar t = T*1.0e-3;
    return (((((
    (
        a6_/6.0*t
        + a5_/5.0)*t
        + a4_/4.0)*t
        + a3_/3.0)*t
        + a2_/2.0)*t
        + a1_)*t
        + a0_*Foam::log(T)
    );
}


Foam::tmp<Foam::scalarField> 
Foam::polyToddYoung::polyIntS(scalarField T)
{
    tmp<scalarField> pField
    (
        new scalarField (T.size(), 1.0)
    );
    scalarField& field = pField.ref();

    scalarField t = T*1.0e-3;
    field = (((((
    (
        a6_/6.0*t
        + a5_/5.0)*t
        + a4_/4.0)*t
        + a3_/3.0)*t
        + a2_/2.0)*t
        + a1_)*t
        + a0_*Foam::log(T)
    );

    return pField;
}


//- Evaluate the definite integral of the polynomial function divided by T
//  from Tlow to Thigh and return the result
//  (used with Cp polynomial for entropy)
//
Foam::scalar Foam::polyToddYoung::polyIntS(scalar Tlow, scalar Thigh)
{
    return(polyIntS(Thigh) - polyIntS(Tlow));
}


Foam::tmp<Foam::scalarField> 
Foam::polyToddYoung::polyIntS(scalar Tlow, scalarField Thigh)
{
    return(polyIntS(Thigh) - polyIntS(Tlow));
}


Foam::tmp<Foam::scalarField> 
Foam::polyToddYoung::polyIntS(scalarField Tlow, scalarField Thigh)
{
    return(polyIntS(Thigh) - polyIntS(Tlow));
}


//- Write the function coefficients
void Foam::polyToddYoung::writeData()
{
    Info<< a0_ << token::SPACE
        << a1_ << token::SPACE
        << a2_ << token::SPACE
        << a3_ << token::SPACE
        << a4_ << token::SPACE
        << a5_ << token::SPACE
        << a6_ << nl;
}


//- Write the function coefficients
void Foam::polyToddYoung::writeData(Ostream& os)
{
    os  << a0_ << token::SPACE
        << a1_ << token::SPACE
        << a2_ << token::SPACE
        << a3_ << token::SPACE
        << a4_ << token::SPACE
        << a5_ << token::SPACE
        << a6_;
}

// ************************************************************************* //
