#!/usr/bin/env python3
"""Parses two given files and computes the l2 error given the file names and the number of doubles"""

import argparse,  logging, os.path
import numpy as np
from matplotlib import pyplot as plt
from decimal import Decimal

def parse_args():
	parser = argparse.ArgumentParser(description = __doc__)
	parser.add_argument("-c", dest="precice_file", help="Mesh file1 (obtained from preCICE)")
	parser.add_argument("-m", dest="monolithic_file", help="Mesh file2 (obtained from chtMultiRegionFoam)")
	parser.add_argument("-n", dest="n_interface_nodes", help="Number of data points in each file")

	args = parser.parse_args()
	return args


def parse_file(file_name, n_lines):
	
	f = open(file_name, "r")
	array = np.zeros(n_lines)

	for i in range(n_lines):
		array[i] = Decimal(f.readline())

	return array

def main():
	args = parse_args()

	# parse the file to a numpy array
	t_precice = parse_file(args.precice_file, int(args.n_interface_nodes))
	t_monolithic = parse_file(args.monolithic_file, int(args.n_interface_nodes))

	# compute error
	theta_precice = (t_precice - 300) / (310 - 300)
	theta_monolithic = (t_monolithic - 300) / (310 - 300)
	error= theta_precice - theta_monolithic
	l2_error = np.sqrt(np.sum((error)**2)/error.size)
	print('l_2 error: {:e} for a total of {} points'.format(l2_error, error.size))

	# ... and plot the error
	offset= 0.5*(1/error.size)
	start= 0 + offset
	end = 1-offset
	x=np.linspace(start, end, num=error.size)
	plt.figure(1)
	plt.plot(x,t_precice, label='coupled')
	plt.plot(x,t_monolithic, label='OpenFOAM-only')
	plt.legend()
	#import tikzplotlib
	#tikzplotlib.save("cht_temperature.tex")

	plt.figure(2)
	plt.plot(x, abs(error), label='error')
	plt.legend()

	plt.show()

if __name__ == "__main__":
	main()
