#!/usr/bin/env python3

import subprocess
import numpy as np
from scipy.fft import fft, fftfreq
from scipy.signal import find_peaks
import matplotlib.pyplot as plt

def compute_fast_fourier_transform(time, quantity, time_step_size, plot = False):
    yf = fft(quantity)
    xf = fftfreq(time.size, time_step_size)[:time.size//2]
    magnitudes = 2.0/time.size * np.abs(yf[0:time.size//2])
    max_index = np.where(magnitudes == np.amax(magnitudes))

    if plot:
        plt.plot(xf, magnitudes)
        plt.show()

    # This might happen if we crop at the wrong place
    if max_index[0][0] == 0:
        # we explicitly exlude the zero frequency
        magnitudes = np.delete(magnitudes, 0)
        max_index = np.where((magnitudes == np.amax(magnitudes)))
        max_index[0][0] += 1

    # the relevant frequency
    return xf[max_index[0][0]]

def compute_mean_and_amp(data, time, min_distance, max_height=None, min_height=None, plot=False):
    max_ind = find_peaks(data, distance=min_distance, height=max_height)
    min_ind = find_peaks(-data, distance=min_distance, height=min_height)

    # We need to average out local max for
    max_val = max(data[max_ind[0][-1]], data[max_ind[0][-2]])
    min_val = min(data[min_ind[0][-1]], data[min_ind[0][-2]])

    if plot:
        plt.plot(time[max_ind[0]], data[max_ind[0]], 'r*')
        plt.plot(time[min_ind[0]], data[min_ind[0]], 'g*')
        plt.plot(time, data)
        plt.show()

    mean = 0.5* (max_val + min_val)
    amp = 0.5* (max_val - min_val)
    return mean, amp

# n_seconds = number of seconds to consider for evaluations
def process_forces(filename, fluid_domain_thickness, time_step_size, n_seconds):
    # First, pipe away the brackets
    command = "sed -i \'s/(/ /g\' {}".format(filename)
    subprocess.run(command, shell=True)
    command = "sed -i \'s/)/ /g\' {}".format(filename)
    subprocess.run(command, shell=True)

    # load the data from the actual file
    force_data = np.loadtxt(filename)

    # pipe away implicit subiterations
    total_iteration = force_data.shape[0]

    # 1. allocate a boolean extractor array
    is_converged = np.zeros((total_iteration, 1))

    # 2. Loop over all preCICE iterations
    for i in range(total_iteration - 2):
        time_index = 0
        current_time = force_data[i][time_index]
        next_time = force_data[ i + 1 ][time_index]

        if current_time < next_time:
            is_converged[i][0] = True

    # The last iteration is always converged
    is_converged[total_iteration - 1][0] = True

    n_converged_iterations = is_converged[:,0].sum()
    print("Number of converged iterations = ", n_converged_iterations)

    # 3. extract the relevant data from the raw array
    indices = np.where(is_converged==True)
    converged_forces = [force_data[index] for index in indices][0]

    # We have now a (n, 10) array, of which the columns 1 and 2 (forces in x and y direction) are relevant

    # Pipe away the initial ramp up by considering the last 5
    n_iterations = n_seconds/time_step_size
    converged_forces = converged_forces[int(n_converged_iterations)-int(n_iterations):int(n_converged_iterations)]

    # Also , we need to scale the forces in order to refer to unit area
    drag = (1/fluid_domain_thickness)*converged_forces[:,1]
    lift = (1/fluid_domain_thickness)*converged_forces[:,2]
    time = converged_forces[:,0]

    # Now compute a Fourier transform in order to evaluate the frequency
    drag_frequency = compute_fast_fourier_transform(time, drag, time_step_size, False)

    # Compute local max and min
    drag_mean, drag_amp = compute_mean_and_amp(drag, time, 0.5*(1./(time_step_size*drag_frequency)), 200, -200, True)

    print("Drag force: {} +- {} with frequency {}".format(drag_mean, drag_amp, drag_frequency))

    # Now compute a Fourier transform in order to evaluate the frequency
    lift_frequency = compute_fast_fourier_transform(time, lift, time_step_size)

    # Compute local max and min
    lift_mean, lift_amp = compute_mean_and_amp(lift, time, 1.*(1./(time_step_size*lift_frequency)), 0,0,True)

    print("Lift force: {} +- {} with frequency {}".format(lift_mean, lift_amp, lift_frequency))


# n_seconds = number of seconds to consider for evaluations
def process_watchpoint(filename, time_step_size, n_seconds):
    # pipe away the headline
    command = "sed -i \'/Time/d\' {}".format(filename)
    subprocess.run(command, shell=True)

    # load the data from the actual file
    data = np.loadtxt(filename)

    total_iteration = data.shape[0]
    # Relevant iterations
    n_iterations = n_seconds/time_step_size
    data = data[int(total_iteration)-int(n_iterations):int(total_iteration)]

    # Also , we need to scale the forces in order to refer to unit area
    x_displ = data[:,3]
    y_displ = data[:,4]
    time = data[:,0]

    # Now compute a Fourier transform in order to evaluate the frequency
    x_frequency = compute_fast_fourier_transform(time, x_displ, time_step_size)

    # Compute local max and min
    x_mean, x_amp = compute_mean_and_amp(x_displ, time, 0.5*(1./(time_step_size*x_frequency)), None, None, True)

    print("x-displacement: {} e-3 +- {} e-3 with frequency {}".format(x_mean*1000, x_amp*1000, x_frequency))

    # Now compute a Fourier transform in order to evaluate the frequency
    y_frequency = compute_fast_fourier_transform(time, y_displ, time_step_size)

    # Compute local max and min
    y_mean, y_amp = compute_mean_and_amp(y_displ, time, 1.0*(1./(time_step_size*y_frequency)), None, None, True)

    print("y-displacement: {} e-3 +- {} e-3 with frequency {}".format(y_mean*1000, y_amp*1000, y_frequency))

def main():
    # Case specific input data
    filename='force.dat'
    fluid_domain_thickness = 0.2 # We need to scale the raw force data
    time_step_size = 0.001
    n_seconds = 5
    process_forces(filename, fluid_domain_thickness, time_step_size, n_seconds)
    process_watchpoint("precice-Solid-watchpoint-Flap-Tip.log", time_step_size, n_seconds)

if __name__ == "__main__":
    main()
